let includeMoviePlus = false;
let autoHideNPlus = false;
let darkMode = false;
let enableVideoEndSound = false;
let visualizeProgress = false;
let discordWebhook = '';
let intervalID = null;
let watchingVideo = false;
let videoEndNotified = false;
let notificationAudio = new Audio(chrome.runtime.getURL('audio/duolingo-completed.mp3'));
let currentVideoTitle = null; // 動画タイトルを保存する変数
let currentVideoSrc = null; // 現在の動画ソースを保存する変数
let lastSyncedData = {}; // 前回同期したデータを保持

// 音声設定
let videoEndSound = 'duolingo-completed.mp3';
let videoEndVolume = 100;
let breakStartSound = 'notify_bright.mp3';
let breakStartVolume = 100;
let breakEndSound = 'timer_end.mp3';
let breakEndVolume = 100;

// 前回ローカルストレージに保存したデータを記録
let lastSavedLocalData = {};

// Global variables to store SVG templates for progress visualization
let testIconTemplate = null; // For incomplete (gray) icons
let testFillIconTemplate = null; // For completed (green) icons

// URL変更検知とサーバーデータキャッシュ用の変数
let contentScriptLastUrl = '';
let cachedCourseHistory = null;
let lastHistoryFetchTime = 0;
const HISTORY_CACHE_DURATION = 5 * 60 * 1000; // 5分間キャッシュ

// デバッグ用の統計情報は削除

// ダークテーママネージャーのインスタンス
let darkThemeManager = null;

/**
 * courses/123/chapters/456 以降のパス要素を切り落として正規化
 * 例: https://.../chapters/456/evaluation_test/999 → https://.../chapters/456
 */
function normalizeCourseUrl(url) {
  const m = url.match(/^https:\/\/www\.nnn\.ed\.nico\/courses\/\d+\/chapters\/\d+/);
  return m ? m[0] : url;
}

// ダークテーママネージャーを初期化
function initializeDarkThemeManager() {
  if (typeof window.DarkThemeManager !== 'undefined') {
    darkThemeManager = new window.DarkThemeManager();
    
    // 初期化完了後、自動検出を実行
    if (darkMode) {
      // 未対応の色を自動検出
      const undetectedColors = darkThemeManager.autoDetectColors();
      if (undetectedColors.length > 0) {
        
        // 自動でダークテーマ色を生成（簡易版）
        const additionalColorMappings = {};
        undetectedColors.forEach(color => {
          const darkColor = generateDarkColor(color);
          if (darkColor) {
            additionalColorMappings[color] = darkColor;
          }
        });
        
        if (Object.keys(additionalColorMappings).length > 0) {
          darkThemeManager.addColorMappings(additionalColorMappings);
        }
      }
    }
  } else {
    // DarkThemeManagerが読み込まれていない場合、少し待ってから再試行
    setTimeout(initializeDarkThemeManager, 100);
  }
}

// 簡易的なダークテーマ色生成関数
function generateDarkColor(originalColor) {
  // rgb(r, g, b) 形式の色を解析
  const rgbMatch = originalColor.match(/rgb\((\d+),\s*(\d+),\s*(\d+)\)/);
  if (!rgbMatch) return null;
  
  const r = parseInt(rgbMatch[1]);
  const g = parseInt(rgbMatch[2]);
  const b = parseInt(rgbMatch[3]);
  
  // 明るい色は暗く、暗い色は明るく変換
  const brightness = (r * 299 + g * 587 + b * 114) / 1000;
  
  if (brightness > 128) {
    // 明るい色 → 暗い色
    const newR = Math.max(0, Math.round(r * 0.2));
    const newG = Math.max(0, Math.round(g * 0.2));
    const newB = Math.max(0, Math.round(b * 0.2));
    return `rgb(${newR}, ${newG}, ${newB})`;
  } else {
    // 暗い色 → 明るい色
    const newR = Math.min(255, Math.round(r * 3 + 100));
    const newG = Math.min(255, Math.round(g * 3 + 100));
    const newB = Math.min(255, Math.round(b * 3 + 100));
    return `rgb(${newR}, ${newG}, ${newB})`;
  }
}

// 初期化
initializeDarkThemeManager();

function updateCountAndTime() {
  const videoElements = document.querySelectorAll('li.sc-aXZVg.sc-gEvEer');
  const testElements = document.querySelectorAll('svg[type="exercise-rounded"]');
  let videoCount = 0;
  let testCount = 0;
  let totalQuestions = 0;
  let totalDuration = 0;
  videoElements.forEach(element => {
    const watchedLabel = element.querySelector('div.sc-aXZVg.JyFzy');
    const icon = element.querySelector('svg.sc-1kjz3de-2');
    const durationElement = element.querySelector('div.sc-aXZVg.iuHQbN');
    if (watchedLabel && watchedLabel.textContent.includes('視聴済み')) {
      return;
    }
    if (icon) {
      const iconType = icon.getAttribute('type');
      // 動画をカウントする条件
      if (iconType === 'movie-rounded' || (iconType === 'movie-rounded-plus' && includeMoviePlus)) {
        const iconStyle = icon.style.color;
        if (iconStyle !== 'rgb(0, 197, 65)') {
          videoCount++;
          // 動画の再生時間を合計する
          if (durationElement) {
            const durationText = durationElement.textContent.trim();
            const [minutes, seconds] = durationText.split(':').map(Number);
            const durationInSeconds = (minutes * 60) + seconds;
            totalDuration += durationInSeconds;
          }
        }
      }
    }
  });
  testElements.forEach(element => {
    const parentLi = element.closest('li.sc-aXZVg.sc-gEvEer');
    const questionCountElement = parentLi.querySelector('div.sc-aXZVg.iFkSEV');
    // テストの進捗が完了していない場合のみカウント
    if (element.getAttribute('fill') === '#00c541' || element.getAttribute('color') === '#00c541') {
      return;
    }
    if (questionCountElement) {
      const questionText = questionCountElement.textContent.trim();
      const questionCount = parseInt(questionText.replace('問', ''), 10);
      if (!isNaN(questionCount)) {
        testCount++;
        totalQuestions += questionCount;
      }
    }
  });

  // URLからコース情報を抽出
  const currentUrl = window.location.href;
  const courseMatch = currentUrl.match(/\/courses\/(\d+)\/chapters\/(\d+)/);
  
  if (courseMatch) {
    const courseInfo = `${courseMatch[1]}/chapters/${courseMatch[2]}`;
    const totalMinutes = Math.floor(totalDuration / 60);
    const totalSeconds = totalDuration % 60;
    
    // コース情報と統計を保存
    chrome.storage.sync.get('courseHistory', function(result) {
      if (chrome.runtime.lastError) {
        console.error('Storage get error:', chrome.runtime.lastError);
        return;
      }
      
      let courseHistory = result.courseHistory || {};
      
      // 新しいデータを作成（簡素化版）
      const newData = {
        url: normalizeCourseUrl(currentUrl),
        title: (document.title || 'コース').substring(0, 50), // タイトルを50文字に制限
        videoTime: `${totalMinutes}分${totalSeconds}秒`,
        videoCount: videoCount,
        testCount: testCount,
        questionCount: totalQuestions,
        timestamp: new Date().toISOString().split('T')[0] // 日付のみ保存（時間は省略）
      };
      
      // 前回保存したデータと比較
      const lastData = lastSavedLocalData[courseInfo];
      const dataChanged = !lastData || 
                         lastData.videoTime !== newData.videoTime ||
                         lastData.videoCount !== newData.videoCount ||
                         lastData.testCount !== newData.testCount ||
                         lastData.questionCount !== newData.questionCount ||
                         lastData.title !== newData.title;
      
      // データが変更された場合のみ保存
      if (dataChanged) {
        
        // 新しいデータを追加
        courseHistory[courseInfo] = newData;
        
        // データ制限：最大20件まで（古いものから削除）
        const entries = Object.entries(courseHistory);
        if (entries.length > 20) {
          // タイムスタンプでソートして古い順に並べる
          entries.sort((a, b) => new Date(a[1].timestamp) - new Date(b[1].timestamp));
          // 最新の20件のみ保持
          const recentEntries = entries.slice(-20);
          courseHistory = Object.fromEntries(recentEntries);
        }
        
        // ストレージサイズをチェック（より厳しい制限）
        const dataSize = JSON.stringify(courseHistory).length;
        if (dataSize > 5000) { // 5KB制限（8KB制限の余裕を持たせる）
          // さらに削減（最大15件）
          const entries = Object.entries(courseHistory);
          entries.sort((a, b) => new Date(a[1].timestamp) - new Date(b[1].timestamp));
          const recentEntries = entries.slice(-15);
          courseHistory = Object.fromEntries(recentEntries);
          
          // それでも大きい場合は最大10件に削減
          const finalSize = JSON.stringify(courseHistory).length;
          if (finalSize > 5000) {
            const entries = Object.entries(courseHistory);
            entries.sort((a, b) => new Date(a[1].timestamp) - new Date(b[1].timestamp));
            const recentEntries = entries.slice(-10);
            courseHistory = Object.fromEntries(recentEntries);
          }
        }
        
        chrome.storage.sync.set({ courseHistory: courseHistory }, function() {
          if (chrome.runtime.lastError) {
            // 8KB制限エラーの場合は緊急クリーンアップを実行
            if (chrome.runtime.lastError.message && 
                chrome.runtime.lastError.message.includes('quota exceeded')) {
              emergencyCleanup();
            }
            
            // エラー時はローカルストレージに保存を試行
            try {
              chrome.storage.local.set({ courseHistory: courseHistory }, function() {
                if (chrome.runtime.lastError) {
                  // Silent fail
                } else {
                  // 前回保存データを更新
                  lastSavedLocalData[courseInfo] = { ...newData };
                }
              });
            } catch (e) {
              // Silent fail
            }
          } else {
            // 前回保存データを更新
            lastSavedLocalData[courseInfo] = { ...newData };
            // 成功時はサーバーに同期
            checkAndSyncToServer(courseInfo, courseHistory[courseInfo]);
          }
        });
      }
    });
  }

  // ストレージから設定を取得して表示する内容を決定
  chrome.storage.sync.get(['showVideoTime', 'showVideoCount', 'showTestCount', 'showQuestionCount'], function(result) {
    if (chrome.runtime.lastError) {
      console.error('Error retrieving settings:', chrome.runtime.lastError);
      return;
    }

    // デフォルト設定を適用（未設定の場合はtrueとする）
    const settings = {
      showVideoTime: result.showVideoTime !== false, // デフォルトtrue
      showVideoCount: result.showVideoCount !== false, // デフォルトtrue
      showTestCount: result.showTestCount !== false, // デフォルトtrue
      showQuestionCount: result.showQuestionCount !== false // デフォルトtrue
    };


    const totalMinutes = Math.floor(totalDuration / 60);
    const totalSeconds = totalDuration % 60;
    const newElement = document.createElement('div');
    newElement.style.marginTop = '10px';

    // 各項目を条件に応じて表示
    if (settings.showVideoTime) {
      newElement.textContent += `動画時間: ${totalMinutes}分${totalSeconds}秒 `;
    }
    if (settings.showVideoCount) {
      newElement.textContent += `動画数: ${videoCount} `;
    }
    if (settings.showTestCount) {
      newElement.textContent += `テスト数: ${testCount} `;
    }
    if (settings.showQuestionCount) {
      newElement.textContent += `問題数: ${totalQuestions}`;
    }
    
    
    const existingElement = document.getElementById('countAndTime');
    const parentElement = document.querySelector('.sc-aXZVg.elbZCm');
    
    if (existingElement) {
      existingElement.innerHTML = `${newElement.outerHTML}`;
    } else {
      if (parentElement) {
        const combinedElement = document.createElement('div');
        combinedElement.id = 'countAndTime';
        combinedElement.innerHTML = `${newElement.outerHTML}`;
        parentElement.appendChild(combinedElement);
      } else {
      }
    }
    
    // コースリンクに残り時間情報を追加
    addRemainingTimeToLinks();
  });

  // Nプラス教材自動非表示の設定が有効な場合、ボタンをクリック
  if (autoHideNPlus) {
    let button1 = Array.from(document.querySelectorAll('button')).find(el => el.textContent === '必修教材のみ');
    let button2 = Array.from(document.querySelectorAll('button')).find(el => el.textContent === 'Nプラス教材のみ');

    let isButton2Selected = false;

    if (button2) {
        // "Nプラス教材のみ"ボタンがクラス名で選択されているかチェック
        const button2Classes = button2.className;
        isButton2Selected = button2Classes.includes('sc-aXZVg bhKrcp sc-13j7nb-0 sc-18fty53-0 sc-nwys29-0 eUGJyl iQRYnJ lePrcr');
    }

    if (button1 && !isButton2Selected) {
        // "必修教材のみ"ボタンがクラス名で選択されているかチェック
        const button1Classes = button1.className;
        const isButton1Selected = button1Classes.includes('sc-aXZVg bhKrcp sc-13j7nb-0 sc-18fty53-0 sc-nwys29-0 eUGJyl iQRYnJ lePrcr');

        // "必修教材のみ"ボタンが選択されていない場合はクリック
        if (!isButton1Selected) {
            button1.click(); // ボタンをクリック
        }
    }
  }

  // ダークモードが有効な場合、applyDarkModeを呼び出す
  if (darkMode && darkThemeManager) {
    darkThemeManager.applyDarkMode();
  }
  
  // 動画視聴完了の検出
  checkVideoCompletion();

  // 進捗の視覚化
  visualizeProgressIcons();
}

// 進捗をアイコンで視覚化する関数
function visualizeProgressIcons() {
  // セレクターをより汎用的にする
  // Find SVG templates if not already found
  if (!testIconTemplate) {
    testIconTemplate = document.querySelector('svg[type="test"]');
  }
  if (!testFillIconTemplate) {
    testFillIconTemplate = document.querySelector('svg[type="test-fill"]');
  }
  if (!testIconTemplate || !testFillIconTemplate) {

  }
  const figures = document.querySelectorAll('figure[class*="sc-hanjyw-2"]');

  figures.forEach(figure => {
    // アイコンとテキスト("6/13")の両方を含むコンテナ
    const container = figure.querySelector('div[class*="sc-fxzfvb-0"]');
    if (!container) return;

    const ariaLabel = figure.getAttribute('aria-label');
    // aria-labelがない、またはレポートの形式でない場合は処理しない
    if (!ariaLabel || !ariaLabel.startsWith('レポート')) {
      return;
    }
    
    const match = ariaLabel.match(/レポート(\d+)個のうち(\d+)個が完了/);

    if (visualizeProgress) {
      // 機能が有効な場合
      if (figure.dataset.visualized === 'true') return;
      
      // レポート形式でない場合は何もしない
      if (!match) return;

      const total = parseInt(match[1], 10);
      const completed = parseInt(match[2], 10);

      // totalが0または数値でない場合は、視覚化しない
      if (isNaN(total) || isNaN(completed) || total === 0) {
        return;
      }

      // Get the original icon from the current figure for fallback if templates are missing
      const originalIconInFigure = container.querySelector('svg');
      if (!originalIconInFigure) return;


      // 元のHTMLとスタイルを保存
      if (!figure.dataset.originalIconPartHTML) {
        figure.dataset.originalIconPartHTML = container.innerHTML;
      }
      if (!figure.dataset.originalIconPartStyle) {
        figure.dataset.originalIconPartStyle = container.getAttribute('style') || '';
      }

      // アイコンとテキスト部分をクリア
      container.innerHTML = '';

      // アイコンコンテナ自体をflexコンテナとして設定
      Object.assign(container.style, {
        display: 'flex',
        flexWrap: 'wrap',
        gap: '2px',
        alignItems: 'center'
      });

      for (let i = 0; i < total; i++) {
        let iconToClone;
        let iconColor;

        if (i < completed) {
          // This icon should be completed (green)
          iconToClone = testFillIconTemplate || originalIconInFigure; // Use test-fill template, fallback to original
          iconColor = 'rgb(0, 197, 65)'; // Green
        } else {
          // This icon should be incomplete (gray)
          iconToClone = testIconTemplate || originalIconInFigure; // Use test template, fallback to original
          iconColor = '#b3b3b3'; // Gray
        }

        const newIcon = iconToClone.cloneNode(true); // Clone the chosen template
        newIcon.style.display = 'block'; // flexアイテムなのでblockでOK
        newIcon.setAttribute('color', iconColor);
        newIcon.querySelector('path')?.setAttribute('fill', iconColor);
        container.appendChild(newIcon);
      }
      figure.dataset.visualized = 'true';
    } else if (figure.dataset.visualized === 'true') {
      // 機能が無効な場合（元の状態に戻す）
      container.innerHTML = figure.dataset.originalIconPartHTML || '';
      container.setAttribute('style', figure.dataset.originalIconPartStyle || '');
      delete figure.dataset.visualized;
      delete figure.dataset.originalIconPartHTML;
      delete figure.dataset.originalIconPartStyle;
    }
  });
}

// 動画視聴完了を検出する関数
function checkVideoCompletion() {
  // メインドキュメントの動画プレーヤーをチェック
  checkVideoCompletionInDocument(document);
  
  // iframe内の動画プレーヤーをチェック
  document.querySelectorAll('iframe').forEach(iframe => {
    try {
      const iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
      checkVideoCompletionInDocument(iframeDocument);
      
      // ネストされたiframeも再帰的にチェック
      checkNestedIframes(iframeDocument);
    } catch (e) {
      // クロスオリジンiframeへのアクセスではエラーが発生することがあるため無視
    }
  });
}

// 指定されたドキュメント内の動画プレーヤーをチェックする関数
function checkVideoCompletionInDocument(doc) {
  const videoPlayers = doc.querySelectorAll('video');
  
  if (videoPlayers.length > 0) {
    // 動画が検出され、まだ視聴中フラグが立っていない場合 (＝動画開始時)
    if (!watchingVideo) {
      watchingVideo = true;
      videoEndNotified = false; // 新しい動画なので通知フラグをリセット
      // 動画開始時にタイトルを取得・保存
      currentVideoTitle = getVideoTitleFromIframe();
      // 取得できなかった場合のフォールバック
      if (!currentVideoTitle) {
        currentVideoTitle = document.title || 'タイトルなし';
      }
    }
    
    videoPlayers.forEach(videoPlayer => {
      // 動画が終了に近い状態かどうかをチェック
      // 設定が初期化されていない場合は、強制的に初期化
      if (!settingsInitialized) {
        initializeSettings();
        return; // 次回のチェックで処理する
      }
      
      // 動画のソースが変わった場合、通知フラグをリセット
      const currentSrc = videoPlayer.src || videoPlayer.currentSrc;
      if (currentSrc && currentSrc !== currentVideoSrc) {
        currentVideoSrc = currentSrc;
        videoEndNotified = false;
      }
      
      if (enableVideoEndSound && 
          videoPlayer.currentTime > 0 && 
          videoPlayer.duration > 0 &&
          videoPlayer.currentTime >= videoPlayer.duration - 0.5 && // 0.5秒前
          !videoEndNotified) {
        
        // カスタム通知音を再生
        playCustomVideoEndSound()
          .then(() => {
            videoEndNotified = true;

            // デスクトップ通知を表示
            if (Notification.permission === 'granted') {
              new Notification('動画が終了しました', {
                body: '次の動画を再生してください。',
                icon: chrome.runtime.getURL('images/notification_icon.png') // アイコンを指定
              });
            } else if (Notification.permission !== 'denied') {
              Notification.requestPermission().then(permission => {
                if (permission === 'granted') {
                  new Notification('動画が終了しました', {
                    body: '次の動画を再生してください。',
                    icon: chrome.runtime.getURL('images/notification_icon.png')
                  });
                }
              });
            }
            
            // Discord Webhookに通知を送信
            sendDiscordNotification();
          })
          .catch(error => {
            console.error('カスタム通知音の再生に失敗しました:', error);
          });
      }
      
      // 動画が始めから再生された場合、通知フラグをリセット
      if (videoPlayer.currentTime < 1) {
        videoEndNotified = false;
      }
    });
  } else {
    // 動画ページから離れた場合（他の動画がない場合）
    if (watchingVideo && !hasVideoInAnyFrame()) {
      watchingVideo = false;
      videoEndNotified = false;
      // 動画視聴が終わったのでタイトルをリセット
      currentVideoTitle = null;
    }
  }
}

// Discord Webhookに通知を送信する関数
function sendDiscordNotification() {
  if (!discordWebhook || discordWebhook.trim() === '') {
    return;
  }
  
  // 現在のページタイトルを取得
  let pageTitle = document.title || '動画視聴';
  let cleanedPageTitle = pageTitle;
  if (pageTitle.includes('- ZEN Study')) {
    cleanedPageTitle = pageTitle.replace('- ZEN Study', '').trim();
  }
  pageTitle = cleanedPageTitle;
  
  // 保存されたタイトルを使用
  const videoTitle = currentVideoTitle;
  
  // 説明文を構築
  let description = '';
  description += `**${pageTitle}**\n`;
  
  if (videoTitle) {
    description += `**${videoTitle}**\n`;
  }
  
  description += '次の動画を再生してください。';
  
  // 現在のURLを取得
  const currentUrl = window.location.href;
  
  // Webhookに送信するデータを作成
  const webhookData = {
    content: '動画視聴が完了しました！',
    embeds: [
      {
        title: '動画視聴完了',
        description: description,
        color: 5814783, // 紫色
        fields: [
          {
            name: 'ページURL',
            value: currentUrl
          },
          {
            name: '完了時刻',
            value: new Date().toLocaleString('ja-JP')
          }
        ],
        footer: {
          text: 'ZEN Study Plus'
        }
      }
    ]
  };
  
  // Fetch APIを使用してWebhookにPOSTリクエストを送信
  fetch(discordWebhook, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    body: JSON.stringify(webhookData)
  })
  .then(response => {
    // 処理成功
  })
  .catch(error => {
    // エラー処理
  });
}

// カスタム動画終了音を再生する関数
async function playCustomVideoEndSound() {
  try {
    // 設定が初期化されていない場合は強制的に初期化
    if (!settingsInitialized) {
      await initializeSettings();
    }
    
    
    // メッセージを通じてオフスクリーンで音声を再生
    await chrome.runtime.sendMessage({
      type: 'playCustomSound',
      soundFile: videoEndSound,
      volume: videoEndVolume / 100
    });
  } catch (error) {
    console.error('カスタム動画終了音の再生に失敗しました:', error);
    // フォールバック: デフォルト音声を再生
    try {
      await notificationAudio.play();
    } catch (fallbackError) {
      console.error('フォールバック音声再生も失敗しました:', fallbackError);
    }
  }
}

// iframe内の特定のヘッダーからタイトルを取得する関数
function getVideoTitleFromIframe() {
  // まず直接ドキュメント内を検索
  const headerElement = document.querySelector('header.sc-aXZVg.iCLTaq h3.sc-1qf3z6b-2.gSFcmh');
  if (headerElement) {
    return headerElement.textContent.trim();
  }
  
  // ドキュメント内に見つからない場合、iframeを検索
  let titleText = null;
  
  function searchInFrames(doc) {
    if (titleText) return; // すでに見つかっている場合は処理をスキップ
    
    doc.querySelectorAll('iframe').forEach(iframe => {
      try {
        if (titleText) return; // すでに見つかっている場合は処理をスキップ
        
        const iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
        const headerElement = iframeDocument.querySelector('header.sc-aXZVg.iCLTaq h3.sc-1qf3z6b-2.gSFcmh');
        
        if (headerElement) {
          titleText = headerElement.textContent.trim();
        } else {
          // 見つからない場合は、さらにネストされたiframeを検索
          searchInFrames(iframeDocument);
        }
      } catch (e) {
        // クロスオリジンiframeへのアクセスではエラーが発生することがあるため無視
      }
    });
  }
  
  searchInFrames(document);
  return titleText;
}

// ネストされたiframeを再帰的にチェックする関数
function checkNestedIframes(doc) {
  doc.querySelectorAll('iframe').forEach(iframe => {
    try {
      const iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
      checkVideoCompletionInDocument(iframeDocument);
      checkNestedIframes(iframeDocument);
    } catch (e) {
      // クロスオリジンiframeへのアクセスではエラーが発生することがあるため無視
    }
  });
}

// いずれかのフレームに動画があるかどうかをチェックする関数
function hasVideoInAnyFrame() {
  // メインドキュメントに動画があるかチェック
  if (document.querySelector('video')) {
    return true;
  }
  
  // iframe内に動画があるかチェック
  let hasVideo = false;
  
  function checkFramesForVideos(doc) {
    doc.querySelectorAll('iframe').forEach(iframe => {
      try {
        const iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
        if (iframeDocument.querySelector('video')) {
          hasVideo = true;
        } else {
          checkFramesForVideos(iframeDocument);
        }
      } catch (e) {
        // クロスオリジンiframeへのアクセスではエラーが発生することがあるため無視
      }
    });
  }
  
  checkFramesForVideos(document);
  return hasVideo;
}

// ログイン状態をチェックしてサーバーに同期する関数
async function checkAndSyncToServer(courseInfo, courseData) {
  try {
    const result = await chrome.storage.local.get('jwt');
    if (typeof result.jwt !== 'string' || result.jwt.length === 0) {
      return; // ログインしていない、またはトークンが不正な場合は何もしない
    }
    
    // 前回同期したデータと比較
    const lastData = lastSyncedData[courseInfo];
    if (lastData && isDataSame(lastData, courseData)) {
      return;
    }
    
    // コースIDとチャプターIDを抽出
    const match = courseInfo.match(/(\d+)\/chapters\/(\d+)/);
    if (!match) {
      return;
    }
    
    // API形式に変換
    const lessonData = {
      course_id: match[1],
      chapter_id: match[2],
      url: normalizeCourseUrl(courseData.url),
      title: courseData.title,
      video_time: courseData.videoTime,
      video_time_s: convertTimeToSeconds(courseData.videoTime),
      video_count: courseData.videoCount,
      test_count: courseData.testCount,
      question_count: courseData.questionCount,
      updated_at: courseData.timestamp
    };
    
    // サーバーに保存
    const response = await fetch("https://zsp-api.yoima.com/saveHistory.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Authorization": "Bearer " + result.jwt,
        "X-Auth-Token": result.jwt
      },
      body: JSON.stringify([lessonData])
    });
    
    if (response.ok) {
      // 同期成功時に前回のデータを更新
      lastSyncedData[courseInfo] = {
        videoTime: courseData.videoTime,
        videoCount: courseData.videoCount,
        testCount: courseData.testCount,
        questionCount: courseData.questionCount
      };
    } else {
      const errorData = await response.json().catch(() => ({ error: '詳細不明' }));
      if (response.status === 401) {
        // トークンが無効なので、ローカルストレージから削除
        await chrome.storage.local.remove(['jwt', 'username']);
        
        // ユーザーに通知
        chrome.notifications.create({
          type: 'basic',
          iconUrl: 'icons/icon48.png',
          title: 'ZSP拡張機能',
          message: '認証の有効期限が切れました。ポップアップから再度ログインしてください。'
        });
      }
    }
  } catch (error) {
    // エラー処理
  }
}

// データが同じかどうかを比較する関数
function isDataSame(oldData, newData) {
  return oldData.videoTime === newData.videoTime &&
         oldData.videoCount === newData.videoCount &&
         oldData.testCount === newData.testCount &&
         oldData.questionCount === newData.questionCount;
}

// 時間文字列を秒に変換する関数
function convertTimeToSeconds(timeString) {
  const match = timeString.match(/(\d+)分(\d+)秒/);
  if (match) {
    return parseInt(match[1]) * 60 + parseInt(match[2]);
  }
  return 0;
}

// 設定変更を受け取る
chrome.runtime.onMessage.addListener(function(message, sender, sendResponse) {
  if (message.type === 'settingChanged') {
    if (message.includeMoviePlus !== undefined) {
      includeMoviePlus = message.includeMoviePlus;
    }
    if (message.visualizeProgress !== undefined) {
      visualizeProgress = message.visualizeProgress;
    }
    updateCountAndTime();
  } else if (message.type === 'autoHideNPlusChanged') {
    autoHideNPlus = message.autoHideNPlus;
    updateCountAndTime();
  } else if (message.type === 'darkModeChanged') {
    darkMode = message.darkMode;
    
    // ダークモードの有効化/無効化
    if (darkMode && darkThemeManager) {
      darkThemeManager.applyDarkMode();
    } else if (!darkMode && darkThemeManager) {
      darkThemeManager.disable();
      // ページをリロードしてオリジナルのスタイルに戻す
      location.reload();
    }
    
    updateCountAndTime();
  } else if (message.type === 'videoEndSoundChanged') {
    enableVideoEndSound = message.enableVideoEndSound;
    // 設定変更時に通知フラグをリセット
    videoEndNotified = false;
  } else if (message.type === 'dynamicIslandSettingChanged') {
    // ダイナミックアイランドの表示設定が変更された場合
    const showDynamicIsland = message.showDynamicIsland;
    
    // グローバル関数としてinitDynamicIslandを呼び出す
    if (typeof window.initDynamicIsland === 'function') {
      window.initDynamicIsland();
    }
  } else if (message.type === 'checkSettingsInitialized') {
    // 設定の初期化状態をポップアップに返す
    sendResponse({ initialized: settingsInitialized });
    return true; // 非同期レスポンスを示す
  } else if (message.type === 'allSettingsLoaded') {
    // ポップアップからの全設定読み込み
    const settings = message.settings;
    
    includeMoviePlus = settings.includeMoviePlus;
    autoHideNPlus = settings.autoHideNPlus;
    darkMode = settings.darkMode;
    enableVideoEndSound = settings.enableVideoEndSound;
    discordWebhook = settings.discordWebhook;
    visualizeProgress = settings.visualizeProgress;
    videoEndSound = settings.videoEndSound;
    videoEndVolume = settings.videoEndVolume;
    breakStartSound = settings.breakStartSound;
    breakStartVolume = settings.breakStartVolume;
    breakEndSound = settings.breakEndSound;
    breakEndVolume = settings.breakEndVolume;
    
    // 設定が完全に読み込まれたことをマーク
    settingsInitialized = true;
    
    // 通知フラグはリセットしない（既に通知済みの動画で再通知を防ぐため）
    
    // ダークモードの適用/解除
    if (darkMode && darkThemeManager) {
      darkThemeManager.applyDarkMode();
    } else if (!darkMode && darkThemeManager) {
      darkThemeManager.disable();
    }
    
    updateCountAndTime();
  } else if (message.type === 'soundSettingsChanged') {
    // 音声設定の更新
    if (message.videoEndSound !== undefined) videoEndSound = message.videoEndSound;
    if (message.videoEndVolume !== undefined) videoEndVolume = message.videoEndVolume;
    if (message.breakStartSound !== undefined) breakStartSound = message.breakStartSound;
    if (message.breakStartVolume !== undefined) breakStartVolume = message.breakStartVolume;
    if (message.breakEndSound !== undefined) breakEndSound = message.breakEndSound;
    if (message.breakEndVolume !== undefined) breakEndVolume = message.breakEndVolume;
  } else if (message.type === 'settingChanged' && message.discordWebhook !== undefined) {
    discordWebhook = message.discordWebhook;
  } else if (message.type === 'playNotificationSound') {
    // タイマー通知音を再生
    try {
      notificationAudio.currentTime = 0; // 音声を最初から再生
      notificationAudio.play().catch(error => {
        // Silent fail
      });
    } catch (error) {
      // Silent fail
    }
  } else if (message.type === 'getDarkThemeStats') {
    // ダークテーマの統計情報を返す
    if (darkThemeManager) {
      const stats = darkThemeManager.getPerformanceInfo();
      sendResponse({
        success: true,
        stats: stats,
        enabled: darkMode
      });
    } else {
      sendResponse({
        success: false,
        message: 'Dark theme manager not initialized'
      });
    }
    return true; // 非同期レスポンスを示す
  }
});

// 設定の初期化状態を管理
let settingsInitialized = false;

// 設定を初期化する関数
function initializeSettings() {
  return new Promise((resolve) => {
    chrome.storage.sync.get(['includeMoviePlus', 'autoHideNPlus', 'darkMode', 'enableVideoEndSound', 'discordWebhook', 'visualizeProgress', 'courseHistory',
                            'videoEndSound', 'videoEndVolume', 'breakStartSound', 'breakStartVolume', 'breakEndSound', 'breakEndVolume'], function(result) {
      if (chrome.runtime.lastError) {
        console.error('Error loading initial settings:', chrome.runtime.lastError);
        resolve();
        return;
      }
      
      includeMoviePlus = result.includeMoviePlus || false;
      autoHideNPlus = result.autoHideNPlus || false;
      darkMode = result.darkMode || false;
      enableVideoEndSound = result.enableVideoEndSound || false;
      discordWebhook = result.discordWebhook || '';
      visualizeProgress = result.visualizeProgress || false;
      
      // 音声設定
      videoEndSound = result.videoEndSound || 'duolingo-completed.mp3';
      videoEndVolume = result.videoEndVolume || 100;
      breakStartSound = result.breakStartSound || 'notify_bright.mp3';
      breakStartVolume = result.breakStartVolume || 100;
      breakEndSound = result.breakEndSound || 'timer_end.mp3';
      breakEndVolume = result.breakEndVolume || 100;
      
      // 既存の履歴データを lastSavedLocalData に読み込む
      const existingHistory = result.courseHistory || {};
      Object.keys(existingHistory).forEach(courseInfo => {
        lastSavedLocalData[courseInfo] = { ...existingHistory[courseInfo] };
      });
      
      settingsInitialized = true;
      resolve();
    });
  });
}

// 初期設定を読み込む
initializeSettings().then(() => {
  updateCountAndTime();

  // 既存のインターバルがあれば解除
  if (intervalID) {
    clearInterval(intervalID);
  }

  // ダークモードの場合は100msごと、それ以外は1000msごとに更新
  intervalID = darkMode 
    ? setInterval(updateCountAndTime, 100) 
    : setInterval(updateCountAndTime, 1000);
  
});

// コースリンクに残り時間情報を表示する関数
async function addRemainingTimeToLinks() {
  // 設定を取得
  chrome.storage.sync.get('showRemainingTime', async function(result) {
    if (chrome.runtime.lastError) {
      console.error('Error getting showRemainingTime setting:', chrome.runtime.lastError);
      return;
    }
    
    // 機能が無効の場合は何もしない
    if (result.showRemainingTime === false) {
      return;
    }
    
    // コース一覧ページのリンクを取得
    const courseLinks = document.querySelectorAll('a[href^="/courses/"]');
    
    if (courseLinks.length === 0) {
      return;
    }
    
    // URL変更を検知
    const currentUrl = window.location.href;
    const urlChanged = currentUrl !== contentScriptLastUrl;
    
    if (urlChanged) {
      contentScriptLastUrl = currentUrl;
      // URL変更時は既存の残り時間要素を削除
      const existingElements = document.querySelectorAll('.remaining-time-indicator');
      existingElements.forEach(elem => elem.remove());
      cachedCourseHistory = null; // キャッシュをクリア
    }
    
    // 既に残り時間要素が存在するリンクがあるかチェック
    const existingElements = document.querySelectorAll('.remaining-time-indicator');
    if (existingElements.length > 0 && !urlChanged) {
      return; // 既に要素が存在し、URLも変更されていない場合はリクエストしない
    }
    
    // キャッシュが有効かチェック
    const now = Date.now();
    const cacheValid = cachedCourseHistory && 
                      (now - lastHistoryFetchTime) < HISTORY_CACHE_DURATION;
    
    let courseHistory = {};
    
    if (cacheValid) {
      // キャッシュを使用
      courseHistory = cachedCourseHistory;
    } else {
      // ログイン状態を確認してサーバーデータとローカルデータを取得
      try {
        // ログイン状態をチェック
        const loginResult = await chrome.storage.local.get('jwt');
        
        if (loginResult.jwt && typeof loginResult.jwt === 'string' && loginResult.jwt.length > 0) {
          // ログインしている場合、オンラインデータを優先して取得
          try {
            const response = await fetch("https://zsp-api.yoima.com/loadHistory.php", {
              headers: { 
                "Authorization": "Bearer " + loginResult.jwt,
                "X-Auth-Token": loginResult.jwt
              }
            });
            
            if (response.ok) {
              const serverData = await response.json();
              // サーバーデータをローカル形式に変換
              serverData.forEach(item => {
                const courseInfo = `${item.course_id}/chapters/${item.chapter_id}`;
                courseHistory[courseInfo] = {
                  url: item.url,
                  title: item.title,
                  videoTime: item.video_time,
                  videoCount: item.video_count,
                  testCount: item.test_count,
                  questionCount: item.question_count,
                  timestamp: item.updated_at,
                  isFromServer: true
                };
              });
            }
          } catch (error) {
            // Silent error
          }
        }
        
        // ローカルデータも取得（オンラインにない情報を補完）
        const localResult = await chrome.storage.sync.get('courseHistory');
        const localHistory = localResult.courseHistory || {};
        
        // オンラインデータを優先し、ローカルデータで補完
        Object.keys(localHistory).forEach(courseInfo => {
          if (!courseHistory[courseInfo]) {
            courseHistory[courseInfo] = {
              ...localHistory[courseInfo],
              isFromServer: false
            };
          }
        });
        
        // キャッシュを更新
        cachedCourseHistory = courseHistory;
        lastHistoryFetchTime = now;
        
      } catch (error) {
        // エラーが発生した場合はローカルデータのみを使用
        const localResult = await chrome.storage.sync.get('courseHistory');
        courseHistory = localResult.courseHistory || {};
        
        // キャッシュを更新
        cachedCourseHistory = courseHistory;
        lastHistoryFetchTime = now;
      }
    }
    
    // コースリンクに残り時間を表示
    courseLinks.forEach(link => {
      // リンクからコースIDとチャプターIDを抽出
      const courseMatch = link.getAttribute('href').match(/\/courses\/(\d+)\/chapters\/(\d+)/);
      if (courseMatch) {
        const courseInfo = `${courseMatch[1]}/chapters/${courseMatch[2]}`;
        
        // 履歴にこのコースの情報があるか確認
        if (courseHistory[courseInfo]) {
          const courseData = courseHistory[courseInfo];
          
          // 残り時間表示用の要素を作成または更新
          let remainingTimeElem = link.querySelector('.remaining-time-indicator');
          
          if (!remainingTimeElem) {
            remainingTimeElem = document.createElement('div');
            remainingTimeElem.className = 'remaining-time-indicator';
            remainingTimeElem.style.fontSize = '12px';
            remainingTimeElem.style.color = '#4A90E2';
            remainingTimeElem.style.fontWeight = 'bold';
            remainingTimeElem.style.marginTop = '5px';
            
            // 進捗バーの下に配置
            const progressContainer = link.querySelector('.sc-1ckq7w0-0');
            if (progressContainer) {
              progressContainer.parentNode.insertBefore(remainingTimeElem, progressContainer.nextSibling);
            } else {
              const contentDiv = link.querySelector('.sc-aXZVg.dKubqp.sc-ni8l2q-1.iUHSzE');
              if (contentDiv) {
                contentDiv.appendChild(remainingTimeElem);
              }
            }
          }
          
          // 残り時間の表示
          const videoTime = courseData.videoTime;
          remainingTimeElem.textContent = `残り時間: ${videoTime}`;
          remainingTimeElem.title = courseData.isFromServer ? 'オンライン同期データ' : 'ローカルデータ';
        }
      }
    });
  });
}

// ストレージ容量管理のユーティリティ関数
function checkStorageQuota() {
  chrome.storage.sync.getBytesInUse(null, function(bytesInUse) {
    if (chrome.runtime.lastError) {
      return;
    }
    
    const totalQuota = chrome.storage.sync.QUOTA_BYTES || 102400; // 100KB
    const usagePercent = (bytesInUse / totalQuota) * 100;
    
    // 80%を超えた場合は警告
    if (usagePercent > 80) {
      // Silent warning
    }
  });
}

// 履歴データのクリーンアップ関数
function cleanupOldHistory() {
  chrome.storage.sync.get('courseHistory', function(result) {
    if (chrome.runtime.lastError) {
      return;
    }
    
    let courseHistory = result.courseHistory || {};
    const entries = Object.entries(courseHistory);
    
    // 30日以上古いデータを削除
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
    
    const filteredEntries = entries.filter(([key, value]) => {
      if (!value.timestamp) return false;
      const entryDate = new Date(value.timestamp);
      return entryDate > thirtyDaysAgo;
    });
    
    // データが減った場合のみ保存
    if (filteredEntries.length < entries.length) {
      courseHistory = Object.fromEntries(filteredEntries);
      chrome.storage.sync.set({ courseHistory: courseHistory }, function() {
        if (chrome.runtime.lastError) {
          // Silent fail
        }
      });
    }
  });
}

// 緊急時のデータクリーンアップ関数
function emergencyCleanup() {
  chrome.storage.sync.get('courseHistory', function(result) {
    if (chrome.runtime.lastError) {
      return;
    }
    
    let courseHistory = result.courseHistory || {};
    const entries = Object.entries(courseHistory);
    
    // 最新の5件のみ保持
    entries.sort((a, b) => new Date(a[1].timestamp) - new Date(b[1].timestamp));
    const recentEntries = entries.slice(-5);
    
    // データをさらに簡素化
    const minimalHistory = Object.fromEntries(
      recentEntries.map(([key, value]) => [
        key,
        {
          videoTime: value.videoTime,
          videoCount: value.videoCount,
          testCount: value.testCount,
          questionCount: value.questionCount,
          timestamp: value.timestamp
        }
      ])
    );
    
    chrome.storage.sync.set({ courseHistory: minimalHistory }, function() {
      if (chrome.runtime.lastError) {
        // 最後の手段：完全にクリア
        chrome.storage.sync.remove('courseHistory', function() {
          // Silent cleanup
        });
      }
    });
  });
}

// 定期的にストレージをチェック（1時間ごと）
setInterval(checkStorageQuota, 60 * 60 * 1000);

// 初回実行
checkStorageQuota();

// 1日1回クリーンアップを実行
const lastCleanup = localStorage.getItem('lastCleanup');
const now = new Date().toDateString();
if (lastCleanup !== now) {
  cleanupOldHistory();
  localStorage.setItem('lastCleanup', now);
}

// DOMContentLoadedイベントで設定の初期化を保証
document.addEventListener('DOMContentLoaded', async function() {
  if (!settingsInitialized) {
    await initializeSettings();
  }
});

// ページ読み込み完了時にも設定を確認
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', async function() {
    if (!settingsInitialized) {
      await initializeSettings();
    }
  });
} else {
  // すでにDOMが読み込まれている場合は即座に実行
  if (!settingsInitialized) {
    initializeSettings();
  }
}
